﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using TMPro;
using UnityEngine;
using UnityEngine.EventSystems;
using Cinemachine;

public class HidingPhase : GamePhase
{
    public float TimeRemaining { get; private set; }
    private List<TimerInterval> _TimerIntervals;

    public override void OnStartCurrentPhase()
    {
        //Enable the hider, setup the timer
        GameController.Instance.HiderPlayerObject.GetComponent<BasicMovement>().CanMove = true;
        GameController.Instance.HiderPlayerObject.GetComponent<SpriteRenderer>().SetOpacity(1.0f);
        TimeRemaining = GameController.Instance.RoundTime;
        PauseMenu.CanPause = true;

        _TimerIntervals = GameController.Instance.TimerIntervals.Select(interval => interval.Clone()).ToList();

        GameController.Instance.Timer.SetActive(true);
        GameController.Instance.Timer.GetComponent<Animator>().speed = 1000.0f / TimeRemaining;
        GameController.Instance.Timer.GetComponent<Animator>().Play("TimerTick", 0, 0);
        GameController.Instance.Timer.GetComponent<Animator>().enabled = true;
        GameController.Instance.IsTimerPaused = false;

        //Hiding, so follow the hider and blend from spawn to the hider player
        GameController.Instance.PlayerVirtualCamera.Follow = GameController.Instance.HiderPlayerObject.transform;
        GameController.Instance.SpawnVirtualCamera.BlendTo(GameController.Instance.PlayerVirtualCamera);
    }

    public override void Update()
    {
        if (!GameController.Instance.IsTimerPaused)
        {
            if (TimeRemaining > 0.0f)
            {
                //We still have time, so decrement by delta time and check if we need to play a tick sound
                TimeRemaining -= (Time.deltaTime * 1000.0f);
                CheckTimerTick();
            }

            else if (TimeRemaining <= 0.0f)
            {
                //Time up, pause the ticking and handle it
                TimeRemaining = 0.0f;
                GameController.Instance.IsTimerPaused = true;
                HandleTimeUp();
            }
        }
    }

    public override void OnEndCurrentPhase()
    {
        //Enable the hider temporarily so we can reset it to idle then disable
        GameController.Instance.HiderPlayerObject.SetActive(true);
        GameController.Instance.HiderPlayerObject.GetComponent<Animator>().Play("Idle", -1, 0.0f);
        GameController.Instance.SeekerPlayerObject.GetComponent<Animator>().Play("Idle", -1, 0.0f);
        GameController.Instance.HiderPlayerObject.SetActive(false);
    }

    /// <summary>
    /// Gets the closest hidable object to the reference game object
    /// </summary>
    /// <param name="referenceObject">The object to reference for the distance calculation</param>
    /// <returns></returns>
    private GameObject GetClosestHidingObject(GameObject referenceObject)
    {
        List<GameObject> hidableObjects = new List<GameObject>();

        //Loop through all game objects that have BoxCollider2D components
        foreach (BoxCollider2D collider in GameController.Instance.GameObjects.GetComponentsInChildren<BoxCollider2D>())
        {
            if (collider.gameObject.tag == Constants.HidableTag)
            {
                //Has the hidable tag so this game object is a collidable hidable
                hidableObjects.Add(collider.gameObject);
            }
        }

        if (hidableObjects.Count > 0)
        {
            GameObject retObj = new GameObject();   //The closest hidable game object
            float shortestDistance = float.MaxValue;    //Shortest distance we've found

            foreach (GameObject gameObject in hidableObjects)
            {
                float distance = Vector2.Distance(gameObject.transform.position, referenceObject.transform.position);   //Calculate distance to this game object

                if (distance < shortestDistance)
                {
                    //Found a closer game object, store it and its distance
                    shortestDistance = distance;
                    retObj = gameObject;
                }
            }

            return retObj;
        }

        else
        {
            Debug.Log("There were no BoxCollider2D hidable objects in the game! Returning null.");
            return null;
        }
    }

    /// <summary>
    /// Called when time runs out, and handles the situation then transitions to the next stage of the game
    /// </summary>
    private void HandleTimeUp()
    {
        if (GameController.Instance.HidingObject == null)
        {
            //We're not in an object, so find the closest one, teleport the hider into it and confirm the location
            GameObject closestHidingObject = GetClosestHidingObject(GameController.Instance.HiderPlayerObject);
            GameController.Instance.HiderPlayerObject.transform.position = closestHidingObject.transform.position;
            GameController.Instance.HiderPlayerObject.GetComponent<BasicMovement>().CanMove = false;
            GameController.Instance.ConfirmHideLocation(closestHidingObject);
        }

        else
        {
            //We were in a location when the time ran out, so confirm it
            GameController.Instance.ConfirmHideLocation();
        }
    }

    /// <summary>
    /// Checks the timer intervals and plays a tick if needed
    /// </summary>
    private void CheckTimerTick()
    {
        if (_TimerIntervals.Count > 0)
        {
            if (_TimerIntervals[0].IntervalValue >= TimeRemaining)
            {
                //We've now past the interval, play the tick
                _TimerIntervals[0].PlayTick();

                if (_TimerIntervals[0].GetType() == typeof(TimerIntervalValue))
                {
                    //If it's an value, just remove it
                    _TimerIntervals.RemoveAt(0);
                }

                else
                {
                    if (((TimerIntervalRange)_TimerIntervals[0]).RemainingValues <= 0)
                    {
                        //If it's a range, remove it only if the remaining values are 0
                        _TimerIntervals.RemoveAt(0);
                    }
                }
            }
        }
    }
}
